// Reach extension popup — exports cookies and shows service health

const SERVICES = {
  'slack.com': {
    name: 'Slack',
    icon: '💬',
    aliases: ['app.slack.com'],
  },
  'atlassian.net': {
    name: 'Jira/Confluence',
    icon: '📋',
    aliases: [],
  },
  'drive.google.com': {
    name: 'Google Drive',
    icon: '📁',
    aliases: ['docs.google.com', 'sheets.google.com', 'slides.google.com'],
  },
};

const SERVICE_HEALTH_MAP = {
  'slack.com': 'slack',
  'atlassian.net': 'confluence',
  'drive.google.com': 'drive',
};

const ALIAS_MAP = {};
for (const [canonical, info] of Object.entries(SERVICES)) {
  ALIAS_MAP[canonical] = canonical;
  for (const alias of info.aliases || []) {
    ALIAS_MAP[alias] = canonical;
  }
}

function getDomain(url) {
  try { return new URL(url).hostname; } catch { return null; }
}

function getServiceForDomain(domain) {
  const canonical = ALIAS_MAP[domain];
  if (canonical) return { canonical, ...SERVICES[canonical] };

  for (const [knownDomain, info] of Object.entries(SERVICES)) {
    if (domain.endsWith('.' + knownDomain) || domain === knownDomain) {
      return { canonical: knownDomain, ...info };
    }
    for (const alias of info.aliases || []) {
      if (domain.endsWith('.' + alias) || domain === alias) {
        return { canonical: knownDomain, ...info };
      }
    }
  }
  return null;
}

async function getConfig() {
  const data = await chrome.storage.sync.get(['apiEndpoint', 'authToken']);
  return {
    apiEndpoint: data.apiEndpoint || '',
    authToken: data.authToken || '',
  };
}

async function getCookiesForDomain(domain) {
  const allCookies = [];
  const domainsToCheck = new Set([domain]);

  const parts = domain.split('.');
  if (parts.length >= 2) {
    domainsToCheck.add(parts.slice(-2).join('.'));
    domainsToCheck.add('.' + parts.slice(-2).join('.'));
  }
  if (parts.length >= 3) {
    domainsToCheck.add(parts.slice(-3).join('.'));
  }

  for (const d of domainsToCheck) {
    try {
      let cookies = await chrome.cookies.getAll({ url: `https://${d.replace(/^\./, '')}` });
      allCookies.push(...cookies);
      cookies = await chrome.cookies.getAll({ domain: d });
      allCookies.push(...cookies);
      if (!d.startsWith('.')) {
        cookies = await chrome.cookies.getAll({ domain: '.' + d });
        allCookies.push(...cookies);
      }
    } catch (e) {
      console.log('Error getting cookies for', d, e);
    }
  }

  const seen = new Set();
  return allCookies.filter(c => {
    const key = `${c.name}|${c.domain}`;
    if (seen.has(key)) return false;
    seen.add(key);
    return true;
  });
}

async function exportCookies(storeDomain, fetchDomain) {
  const config = await getConfig();
  if (!config.apiEndpoint || !config.authToken) {
    throw new Error('Not configured. Click "Configure API endpoint" below.');
  }

  const cookies = await getCookiesForDomain(fetchDomain);
  if (cookies.length === 0) {
    throw new Error('No cookies found. Make sure you\'re logged in.');
  }

  const cookieData = cookies.map(c => ({
    name: c.name,
    value: c.value,
    domain: c.domain,
    path: c.path,
    secure: c.secure,
    httpOnly: c.httpOnly,
    sameSite: c.sameSite || 'Lax',
    expires: c.expirationDate,
  }));

  let response;
  try {
    response = await fetch(`${config.apiEndpoint}/cookies/${encodeURIComponent(storeDomain)}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${config.authToken}`,
      },
      body: JSON.stringify({ cookies: cookieData }),
    });
  } catch (e) {
    throw new Error(`Can't reach API — check endpoint in options`);
  }

  if (!response.ok) {
    const text = await response.text();
    throw new Error(`API error (${response.status}): ${text.slice(0, 100)}`);
  }

  return await response.json();
}

function formatAge(isoDate) {
  if (!isoDate) return { text: 'Unknown', stale: true };
  try {
    const stored = new Date(isoDate);
    const now = new Date();
    const diffMs = now - stored;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);

    if (diffDays > 7) return { text: `${diffDays}d ago`, stale: true };
    if (diffDays >= 1) return { text: `${diffDays}d ago`, stale: false };
    if (diffHours >= 1) return { text: `${diffHours}h ago`, stale: false };
    if (diffMins >= 1) return { text: `${diffMins}m ago`, stale: false };
    return { text: 'Just now', stale: false };
  } catch {
    return { text: 'Unknown', stale: true };
  }
}

async function updateUI() {
  const exportBtn = document.getElementById('exportBtn');
  const tabDomain = document.getElementById('tabDomain');
  const tabStatus = document.getElementById('tabStatus');
  const storedCookies = document.getElementById('storedCookies');

  const config = await getConfig();

  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  const domain = getDomain(tab?.url);

  if (!domain) {
    tabDomain.textContent = 'No domain';
    tabStatus.textContent = 'Cannot export from this page';
    exportBtn.disabled = true;
    return;
  }

  const service = getServiceForDomain(domain);

  if (service) {
    tabDomain.textContent = `${service.icon} ${service.name}`;
    tabStatus.textContent = domain !== service.canonical
      ? `${domain} → ${service.canonical}`
      : domain;
    exportBtn.disabled = false;
    exportBtn.onclick = () => handleExport(service.canonical, domain);
  } else {
    tabDomain.textContent = domain;
    tabStatus.textContent = 'Not a recognized service (can still export)';
    exportBtn.disabled = false;
    exportBtn.onclick = () => handleExport(domain, domain);
  }

  if (!config.apiEndpoint) {
    storedCookies.innerHTML = '<div class="empty">Not configured yet — set API endpoint in options</div>';
    return;
  }

  storedCookies.innerHTML = Object.entries(SERVICES).map(([canonical, info]) => `
    <div class="cookie-item">
      <span class="cookie-domain">${info.icon} ${info.name}</span>
      <span class="cookie-status checking">…</span>
    </div>
  `).join('');

  fetchServiceHealth(config);
}

async function fetchServiceHealth(config) {
  try {
    const resp = await fetch(`${config.apiEndpoint}/cookies/health`, {
      headers: {
        'Authorization': `Bearer ${config.authToken}`,
        'Accept': 'application/json',
      },
    });

    if (!resp.ok) {
      showFallbackStatus();
      return;
    }

    const data = await resp.json();
    const services = data.services || {};

    const items = document.querySelectorAll('.cookie-item');
    items.forEach(item => {
      const nameEl = item.querySelector('.cookie-domain');
      const statusEl = item.querySelector('.cookie-status');
      if (!nameEl || !statusEl) return;

      const text = nameEl.textContent;
      let healthKey = null;
      for (const [canonical, key] of Object.entries(SERVICE_HEALTH_MAP)) {
        if (text.includes(SERVICES[canonical].name)) {
          healthKey = key;
          break;
        }
      }

      if (!healthKey || !services[healthKey]) {
        statusEl.className = 'cookie-status missing';
        statusEl.textContent = '—';
        return;
      }

      const info = services[healthKey];
      if (info.connected) {
        const age = formatAge(info.stored_at);
        if (age.stale) {
          statusEl.className = 'cookie-status stale';
          statusEl.textContent = `⚠ ${info.cookie_count || 0} · ${age.text}`;
        } else {
          statusEl.className = 'cookie-status ok';
          statusEl.textContent = `✓ ${info.cookie_count || 0} · ${age.text}`;
        }
      } else {
        statusEl.className = 'cookie-status missing';
        statusEl.textContent = '—';
      }
    });
  } catch {
    showFallbackStatus();
  }
}

function showFallbackStatus() {
  document.querySelectorAll('.cookie-status').forEach(el => {
    if (el.classList.contains('checking')) {
      el.className = 'cookie-status missing';
      el.textContent = '—';
    }
  });
}

async function handleExport(storeDomain, fetchDomain) {
  const exportBtn = document.getElementById('exportBtn');
  const originalText = exportBtn.textContent;

  exportBtn.disabled = true;
  exportBtn.textContent = 'Exporting...';
  exportBtn.classList.remove('success', 'error');

  try {
    const result = await exportCookies(storeDomain, fetchDomain);
    exportBtn.textContent = `Stored ${result.cookie_count} cookies`;
    exportBtn.classList.add('success');
    setTimeout(() => updateUI(), 500);
  } catch (error) {
    exportBtn.textContent = error.message.slice(0, 35);
    exportBtn.classList.add('error');
  }

  setTimeout(() => {
    exportBtn.textContent = originalText;
    exportBtn.classList.remove('success', 'error');
    exportBtn.disabled = false;
  }, 2500);
}

document.getElementById('openOptions')?.addEventListener('click', () => {
  chrome.runtime.openOptionsPage();
});

updateUI();
